-- FS25_GateFix: Allow opening map gates while on contracts

local GateFix = {
    hasPatchedAccessHandler = false,
    hasPatchedAnimatedObjects = false,
    hasPatchedAnimatedActivatable = false,
    originalCanFarmAccess = nil,
    originalCanFarmAccessLand = nil,
    originalGetCanTriggerAnimatedObject = nil,
    originalAnimatedObjectGetIsActivatable = nil
}

-- Utility: try to get a representative world position for an object
local function getObjectWorldXZ(obj)
    if obj == nil then
        return nil, nil
    end

    local node = nil
    -- Common fields across objects
    node = obj.nodeId or obj.triggerNode or obj.rootNode or obj.root or node
    if node == nil and obj.components ~= nil and obj.components[1] ~= nil then
        node = obj.components[1].node
    end

    if node ~= nil and node ~= 0 then
        local x, _, z = getWorldTranslation(node)
        return x, z
    end

    return nil, nil
end

local function getAnimatedObjectWorldXZ(animatedObject)
    if animatedObject == nil then
        return nil, nil
    end

    local node = animatedObject.triggerNode or animatedObject.nodeId
    if node ~= nil and node ~= 0 then
        local x, _, z = getWorldTranslation(node)
        return x, z
    end

    return nil, nil
end

local missionUsesFarmland
local hasContractAccessAtPosition
local isMissionFarmAtPosition
local hasMissionAccessToAnimatedObject

missionUsesFarmland = function(mission, farmlandId)
    if mission == nil or farmlandId == nil then
        return false
    end

    local field = mission.field
    if field ~= nil then
        local farmland = field.farmland
        if farmland ~= nil then
            local fieldFarmlandId = farmland.id
            if fieldFarmlandId == nil and farmland.getId ~= nil then
                fieldFarmlandId = farmland:getId()
            end
            if fieldFarmlandId == farmlandId then
                return true
            end
        end
    end

    if mission.farmlandId ~= nil and mission.farmlandId == farmlandId then
        return true
    end

    if mission.spot ~= nil and mission.spot.farmlandId ~= nil and mission.spot.farmlandId == farmlandId then
        return true
    end

    if mission.spots ~= nil then
        for _, spot in pairs(mission.spots) do
            if type(spot) == "table" then
                local spotFarmlandId = spot.farmlandId
                if spotFarmlandId == nil and spot.getFarmlandId ~= nil then
                    spotFarmlandId = spot:getFarmlandId()
                end
                if spotFarmlandId == farmlandId then
                    return true
                end
            end
        end
    end

    return false
end

hasContractAccessAtPosition = function(farmId, x, z)
    if farmId == nil or x == nil or z == nil then
        return false
    end
    return isMissionFarmAtPosition(farmId, x, z)
end

isMissionFarmAtPosition = function(farmId, x, z)
    if farmId == nil or g_missionManager == nil or x == nil or z == nil then
        return false
    end

    local mission = g_missionManager:getMissionAtWorldPosition(x, z)
    if mission ~= nil and mission.farmId == farmId then
        return true
    end

    if g_farmlandManager == nil then
        return false
    end

    local farmlandId = g_farmlandManager:getFarmlandIdAtWorldPosition(x, z)
    if farmlandId == nil then
        return false
    end

    local missions = g_missionManager.missions
    if missions == nil then
        return false
    end

    for _, activeMission in ipairs(missions) do
        if activeMission ~= nil and activeMission.farmId == farmId then
            if missionUsesFarmland(activeMission, farmlandId) then
                return true
            end
        end
    end

    return false
end

hasMissionAccessToAnimatedObject = function(animatedObject, farmId)
    if animatedObject == nil or farmId == nil then
        return false
    end

    local x, z = getAnimatedObjectWorldXZ(animatedObject)
    if x ~= nil and z ~= nil then
        return isMissionFarmAtPosition(farmId, x, z)
    end

    return false
end

-- Apply overrides once AccessHandler is available (dedicated servers may load mods before the engine files)
local function tryPatchAccessHandler()
    if GateFix.hasPatchedAccessHandler then
        return
    end

    if AccessHandler == nil or AccessHandler.canFarmAccess == nil or AccessHandler.canFarmAccessLand == nil then
        return
    end

    GateFix.originalCanFarmAccess = GateFix.originalCanFarmAccess or AccessHandler.canFarmAccess
    GateFix.originalCanFarmAccessLand = GateFix.originalCanFarmAccessLand or AccessHandler.canFarmAccessLand

    local originalCanFarmAccess = GateFix.originalCanFarmAccess
    local originalCanFarmAccessLand = GateFix.originalCanFarmAccessLand

    -- Extend: AccessHandler.canFarmAccess
    -- If base denies access, allow when the position lies on an active mission for the requesting farm
    function AccessHandler.canFarmAccess(self, farmId, object, mustBeOwner)
        local ok = originalCanFarmAccess(self, farmId, object, mustBeOwner)
        if ok then
            return true
        end

        if mustBeOwner then
            return false
        end

        if farmId == nil or g_missionManager == nil then
            return false
        end

        local x, z = getObjectWorldXZ(object)
        if hasContractAccessAtPosition(farmId, x, z) then
            return true
        end

        return false
    end

    -- Extend: AccessHandler.canFarmAccessLand
    -- If base denies access, allow when the position lies on an active mission for the requesting farm
    function AccessHandler.canFarmAccessLand(self, farmId, x, z, ignoreContract)
        local ok = originalCanFarmAccessLand(self, farmId, x, z, ignoreContract)
        if ok then
            return true
        end

        if ignoreContract then
            return false
        end

        if farmId == nil or g_missionManager == nil or x == nil or z == nil then
            return false
        end

        if hasContractAccessAtPosition(farmId, x, z) then
            return true
        end

        return false
    end

    GateFix.hasPatchedAccessHandler = true
end

local function tryPatchAnimatedObjects()
    if GateFix.hasPatchedAnimatedObjects then
        return
    end

    if PlaceableAnimatedObjects == nil or PlaceableAnimatedObjects.getCanTriggerAnimatedObject == nil then
        return
    end

    GateFix.originalGetCanTriggerAnimatedObject = GateFix.originalGetCanTriggerAnimatedObject or PlaceableAnimatedObjects.getCanTriggerAnimatedObject
    local originalGetCanTriggerAnimatedObject = GateFix.originalGetCanTriggerAnimatedObject

    function PlaceableAnimatedObjects.getCanTriggerAnimatedObject(placeable, animatedObject)
        local allowed = originalGetCanTriggerAnimatedObject(placeable, animatedObject)
        if allowed then
            return true
        end

        if g_currentMission == nil then
            return false
        end

        local farmId = g_currentMission:getFarmId()
        if farmId == nil then
            return false
        end

        local x, z = getAnimatedObjectWorldXZ(animatedObject)
        if hasContractAccessAtPosition(farmId, x, z) then
            return true
        end

        return false
    end

    GateFix.hasPatchedAnimatedObjects = true
end

local function tryPatchAnimatedActivatable()
    if GateFix.hasPatchedAnimatedActivatable then
        return
    end

    if AnimatedObjectActivatable == nil or AnimatedObjectActivatable.getIsActivatable == nil then
        return
    end

    GateFix.originalAnimatedObjectGetIsActivatable = GateFix.originalAnimatedObjectGetIsActivatable or AnimatedObjectActivatable.getIsActivatable
    local originalActivatableGetIsActivatable = GateFix.originalAnimatedObjectGetIsActivatable

    function AnimatedObjectActivatable.getIsActivatable(self, ...)
        local ok = originalActivatableGetIsActivatable(self, ...)
        if ok then
            return true
        end

        if g_currentMission == nil then
            return false
        end

        local farmId = g_currentMission:getFarmId()
        if farmId == nil then
            return false
        end

        if hasMissionAccessToAnimatedObject(self.animatedObject, farmId) then
            return true
        end

        return false
    end

    GateFix.hasPatchedAnimatedActivatable = true
end

local function tryApplyOverrides()
    tryPatchAccessHandler()
    tryPatchAnimatedObjects()
    tryPatchAnimatedActivatable()
end

-- Ensure the override runs when the mission starts (server + clients)
local GateFixEventListener = {}

function GateFixEventListener:loadMap()
    tryApplyOverrides()
end

function GateFixEventListener:onMissionInitialize()
    tryApplyOverrides()
end

function GateFixEventListener:update(dt)
    tryApplyOverrides()
end

addModEventListener(GateFixEventListener)

-- Attempt an early patch in case the engine has already loaded AccessHandler
tryApplyOverrides()

return GateFix
